CREATE TABLE cycles (
    cyc_id               serial PRIMARY KEY,
    cyc_active           boolean NOT NULL DEFAULT TRUE,          -- Anwenderoption
    cyc_done             boolean NOT NULL DEFAULT FALSE,         -- intern, automatisches Schließen/Öffnen
    cyc_src_dbrid        varchar(32) NOT NULL,                   -- zug. Datensatz
    cyc_src_key          varchar(100) NOT NULL,                  -- Schlüssel für Funktionen
    cyc_trg_table        varchar(100),                           -- Zieltabelle für bestimmte Erzeugungsfunktionen
    cyc_begin            date,                                   -- Ausführungsbeginn
    cyc_end              date,                                   -- Ausführungsende
    cyc_nexterm          date,                                   -- nächster errechneter Termin
    cyc_lasterm          date,                                   -- letzter Termin einer ausgeführten Aktion
    cyc_interval         interval,                               -- Intervall-Art (Tag, Woche, Monat, Jahr)
    cyc_multi            integer,                                -- Menge des Intervalls
    cyc_float            integer NOT NULL DEFAULT 14,            -- Vorlauzeit (Tage), nextexec = date + multi * interval - float*'1 day'::interval
    CHECK (cyc_multi > 0),
    CHECK (cyc_float >= 0),
    -- beide gefüllt oder beide NULL
    CONSTRAINT cyc_interval_valid CHECK (cyc_multi IS NULL AND cyc_interval IS NULL OR cyc_multi IS NOT NULL AND cyc_interval IS NOT NULL)
  );

  -- Das nächste Ausführungsdatum anhand Start, Ende, Intervall setzen
  CREATE OR REPLACE FUNCTION cycles__a_iu_nexterm() RETURNS TRIGGER AS $$
    BEGIN
      -- geht erst nach AFTER weil die Funktion den Satz aus cycles braucht.
      UPDATE cycles SET
        cyc_nexterm = cycles__date_next__by__cycle_dbrid(new.cyc_begin, new.cyc_end, new.cyc_lasterm, new.cyc_interval, new.cyc_multi, new.dbrid),
        cyc_done = NOT (new.cyc_end IS NULL OR new.cyc_end >= current_date)
      WHERE cyc_id = new.cyc_id;

      RETURN new;
    END $$ LANGUAGE plpgsql;
    --

    CREATE TRIGGER cycles__a_iu_nexterm
      AFTER INSERT OR UPDATE OF cyc_begin, cyc_end, cyc_interval, cyc_multi
      ON cycles
      FOR EACH ROW
      EXECUTE PROCEDURE cycles__a_iu_nexterm();
--

-- Das nächste Ausführungsdatum berechnen (nur Anhand der übergebenen Zeitwerte)
CREATE OR REPLACE FUNCTION cycles__date_next__calculate(IN inBeginDate date, IN inEndDate date, IN inLastDate date, IN inInterval interval, IN inMulti integer) RETURNS date AS $$
  BEGIN
    WHILE inBeginDate < current_date LOOP -- laut Intervall bis nach heute durchrechnen
        inBeginDate:=inBeginDate + inInterval * inMulti;
    END LOOP;

    WHILE inBeginDate <= inLastDate LOOP -- gab es schon eine Ausführung in der Vergangenheit oder heute, dann Intervalle drauf
        inBeginDate:=inBeginDate + inInterval * inMulti;
    END LOOP;

    -- wenn nächster Termin über angegebenem Ende
    IF inBeginDate > inEndDate THEN
        inBeginDate:=NULL;
    END IF;

    -- Ansonsten ist das neue Datum auch das Rückgabedatum, wenn >= current_date bzw. NULL
    RETURN inBeginDate;
  END $$ LANGUAGE plpgsql;
--

-- Das nächste Ausführungsdatum berechnen (Kopfdaten der entsprechenden Tabelle/Key mit einbeziehen)
CREATE OR REPLACE FUNCTION cycles__date_next__by__cycle_dbrid(IN inBeginDate date, IN inEndDate date, IN inLastDate date, IN inInterval interval, IN inMulti integer, IN inPosDbrid varchar(32)) RETURNS date AS $$
  DECLARE src_key varchar(100);
          headerData record;
  BEGIN
    SELECT cyc_src_key INTO src_key FROM cycles WHERE cycles.dbrid = inPosDbrid;

    IF src_key = 'vertrag_pos' THEN
        -- Anfangs- und Endedatum aus Vertragskopf
        SELECT vtr_beginn AS head_begin, vtr_ende AS head_end INTO headerData
          FROM cycles
          JOIN vertrag_pos ON vertrag_pos.dbrid = cyc_src_dbrid
          JOIN vertrag ON vtr_nr = vtp_vtr_nr
         WHERE cycles.dbrid = inPosDbrid;
    -- ELSIF src_key = 'was_anderes' THEN hole andere Subdaten
    ELSE
        -- leerer Satz für einmaligen Funktionsaufruf (s.u.), ELSIF Erweiterung möglich
        SELECT NULL::date AS head_begin, NULL::date AS head_end INTO headerData;
    END IF;

    -- GREATEST und LEAST für Abgleich der Positionsdatumse innerhalb der Grenzen des Anfang-Ende-Intervalls vom Vertragskopf
    -- (Vertragsbeginn, ..., Vertragsende) führendes Intervall, Funktionen geben nur NULL zurück wenn alles NULL
    RETURN cycles__date_next__calculate(GREATEST(inBeginDate, headerData.head_begin), LEAST(inEndDate, headerData.head_end), inLastDate, inInterval, inMulti);
  END $$ LANGUAGE plpgsql;
--

-- Zyklus-interval (multi|interval|float) in einen String umwandeln (z.B. zum einfachen anzeigen JOINen in Tabellen, wo Zyklen angehangen sind, ohne die Feld-Übersetzung überall einbauen zu müssen)
CREATE OR REPLACE FUNCTION cycle_to_str(IN inMulti integer, IN inInterval interval, IN inFloat integer DEFAULT NULL) RETURNS varchar(50) AS $$
  DECLARE str varchar;
  BEGIN
    CASE
        WHEN inInterval = '1 days '  AND inMulti = 1 THEN str := lang_text(16217);  -- Jeden Tag
        WHEN inInterval = '1 days '                  THEN str := lang_text(10443) || ' ' || inMulti || ' ' || lang_text(749);   -- Alle x Tage
        WHEN inInterval = '7 days '  AND inMulti = 1 THEN str := lang_text(16218);  -- Jede Woche
        WHEN inInterval = '7 days '                  THEN str := lang_text(10443) || ' ' || inMulti || ' ' || lang_text(821);   -- Alle x Wochen
        WHEN inInterval = '1 mons '  AND inMulti = 1 THEN str := lang_text(16219);  -- Jeden Monat
        WHEN inInterval = '1 mons '                  THEN str := lang_text(10443) || ' ' || inMulti || ' ' || lang_text(15334); -- Alle x Monate
        WHEN inInterval = '1 years ' AND inMulti = 1 THEN str := lang_text(16220);  -- Jedes Jahr
        WHEN inInterval = '1 years '                 THEN str := lang_text(10443) || ' ' || inMulti || ' ' || lang_text(2895);  -- Alle x Jahre
        ELSE                                              str := inMulti || ' (error)';
    END CASE;

    IF inFloat IS NOT NULL THEN
        IF inFloat = 1 THEN
            str := str || ' (' || lang_text(16221) || ': 1 ' || lang_text(3027) || ')';                -- Vorlauf: 1 Tag
        ELSEIF inFloat > 0 THEN
            str := str || ' (' || lang_text(16221) || ': ' || inFloat || ' ' || lang_text(749) || ')'; -- Vorlauf: x Tage
        END IF;
    END IF;

    RETURN str;
  END $$ LANGUAGE plpgsql;
--

--
CREATE OR REPLACE FUNCTION cycle_to_str(IN inID BIGINT) RETURNS varchar(50) AS $$
  BEGIN
    RETURN cycle_to_str(cyc_multi, cyc_interval, cyc_float) FROM cycles WHERE cyc_id = inID;
  END $$ LANGUAGE plpgsql;
--

-- Active von TableName + DBRID
CREATE OR REPLACE FUNCTION cycle_isactive(IN tname varchar(100), IN tdbrid varchar(32)) RETURNS boolean AS $$
  BEGIN
    RETURN cyc_active FROM cycles WHERE cyc_src_key = tname AND cyc_src_dbrid = tdbrid;
  END $$ LANGUAGE plpgsql;
--

-- prüfen, ob der Zyklus verwendet wird (TRUE=ja  |  FALSE=nein, DBRID in Tabelle nicht gefunden  |  NULL=Tabelle existiert nicht)
CREATE OR REPLACE FUNCTION cycle__isused(IN inTable varchar(100), IN inDBRID varchar(32)) RETURNS BOOL AS $$
  DECLARE res BOOL;
  BEGIN
    IF NOT EXISTS (SELECT true FROM pg_class WHERE relname ILIKE inTable) THEN
        RETURN NULL;
    END IF;
    EXECUTE 'SELECT EXISTS(SELECT TRUE FROM ' || quote_ident(inTable) || ' WHERE dbrid = ' || quote_literal(inDBRID) || ')' INTO res;
    RETURN res;
  END $$ LANGUAGE plpgsql;
--

--
CREATE OR REPLACE FUNCTION cycle__isused(IN inID BIGINT) RETURNS BOOL AS $$
  BEGIN
    RETURN cycle__isused(cyc_src_key, cyc_src_dbrid) FROM cycles WHERE cyc_id = inID;
  END $$ LANGUAGE plpgsql;
--

-- Zeiteinheiten für cyc_interval (E=EmptyValue D=Tage W=Wochen M=Monate Y=Jahre)
CREATE OR REPLACE FUNCTION cycle_intervals(IncludeValues varchar DEFAULT 'DWMY') RETURNS TABLE(idx INT, interv varchar(10), descr varchar(20)) AS $$
    SELECT 0, NULL       AS interv, ''               AS descr  --(leer)
    WHERE position('E' in upper($1)) > 0
    UNION
    SELECT 1, '1 days '  AS interv, lang_text(749)   AS descr  --Tage
    WHERE position('D' in upper($1)) > 0
    UNION
    SELECT 2, '7 days '  AS interv, lang_text(821)   AS descr  --Wochen
    WHERE position('W' in upper($1)) > 0
    UNION
    SELECT 3, '1 mons '  AS interv, lang_text(15334) AS descr  --Monate
    WHERE position('M' in upper($1)) > 0
    UNION
    SELECT 4, '1 years ' AS interv, lang_text(2895)  AS descr  --Jahre
    WHERE position('Y' in upper($1)) > 0
    ORDER BY 1
  $$ LANGUAGE SQL;
--

-- Ziele für cyc_trg_table
CREATE OR REPLACE FUNCTION cycle_targets() RETURNS TABLE(idx INT, target varchar(30), descr varchar(50)) AS $$
    SELECT 0, NULL         AS target, lang_text(17410) AS descr  --An dieser Stelle
    UNION
    SELECT 1, 'qabservice' AS target, lang_text(17411) AS descr  --Neuer Servicevorfall
    ORDER BY 1
  $$ LANGUAGE SQL;
--

-- CREATE OR REPLACE VIEW cycles_info AS
    --  SELECT ... in ViewsAnd Rules verlagert damit die Tabellen da sind.


-- Zyklus Funktionen (thematisch verschienden, funktionell hier zusammengehalten)

-- Vertragsautomatismen (Kopfdaten)
CREATE OR REPLACE FUNCTION vertrag__ende_done__auto() RETURNS VOID AS $$
  BEGIN
    -- automatische Vertragsverlängerung
    UPDATE vertrag
       SET vtr_ende = vtr_ende + vtr_laufverl_interval * vtr_laufverl
     WHERE NOT vtr_done
       AND vtr_autolaufverl
       AND vtr_ende - '14 days'::interval <= current_date;

    -- automatisch Beenden
    UPDATE vertrag
       SET vtr_done = TRUE
     WHERE NOT vtr_done
       AND NOT vtr_autolaufverl
       AND vtr_ende < current_date;

    RETURN;
  END $$ LANGUAGE plpgsql;
--

-- zentrale Zyklus Funktion
CREATE OR REPLACE FUNCTION cycles__execute() RETURNS VOID AS $$
  DECLARE _cyclesrec record;
          _today date;
  BEGIN
    _today := current_date; -- mehrfache Abfrage verhindern

    -- automatisches Löschen der nicht mehr verwendeten
    DELETE FROM cycles
     WHERE NOT cycle__isused(cyc_id); -- macht ein EXISTS(SELECT TRUE FROM ' || quote_ident(cyc_src_key) || ' WHERE dbrid = ' || quote_literal(cyc_src_dbrid) || ')

    -- automatisch Beenden
    UPDATE cycles
       SET cyc_done = TRUE
     WHERE
       NOT cyc_done
       AND cyc_end < _today;

    -- nexterm die in Vergangenheit liegen neu berechnen. Kann eigtl. nur vorkommen, wenn Zyklus nicht regelmäßig ausgelöst wurde.
    UPDATE cycles SET
           cyc_nexterm = cycles__date_next__by__cycle_dbrid(cyc_begin, cyc_end, cyc_lasterm, cyc_interval, cyc_multi, cycles.dbrid)
     WHERE
       NOT cyc_done
       AND cyc_nexterm < _today - '1 day'::interval; -- Ausnahme gestern: Erstellung und Termin gestern, Ausführung ist erst nach 1:00 am nächsten Tag

    FOR _cyclesrec IN
       SELECT cyc_id, cyc_src_dbrid, cyc_src_key, cyc_nexterm, cyc_interval, cyc_multi, cyc_float, cyc_trg_table
         FROM cycles
        WHERE cyc_active
          AND NOT cyc_done
          AND cyc_nexterm - COALESCE(cyc_float, 0) * '1 day'::interval <= _today
        ORDER BY cyc_src_key, cyc_nexterm, cyc_id LOOP

        -- While Fall: Erstellung mit Intervall mehrmals innerhalb Vorhaltezeit ab gestern. Muss in die Zukunft angelegt werden bis Vorhaltezeit erreicht.
        WHILE _cyclesrec.cyc_nexterm - COALESCE(_cyclesrec.cyc_float, 0) * '1 day'::interval <= _today LOOP

            BEGIN
                 IF _cyclesrec.cyc_src_key = 'vertrag_pos' THEN
                     PERFORM cycle__execute__vertrag_pos(_cyclesrec.cyc_src_dbrid);
                 END IF;

                 IF _cyclesrec.cyc_src_key = 'ab2' THEN
                     PERFORM cycle__execute__ab2(_cyclesrec.cyc_src_dbrid);
                 END IF;

                 IF _cyclesrec.cyc_src_key = 'abk' THEN
                     PERFORM cycle__execute__abk(_cyclesrec.cyc_src_dbrid, _cyclesrec.cyc_trg_table);
                 END IF;

                 -- UPDATE nötig, da cycles-Daten in Aktions-Funktionen gebraucht werden
                 UPDATE cycles
                    SET cyc_lasterm = cyc_nexterm,
                        cyc_nexterm = cycles__date_next__by__cycle_dbrid(cyc_begin, cyc_end, cyc_nexterm, cyc_interval, cyc_multi, cycles.dbrid)
                  WHERE cyc_id = _cyclesrec.cyc_id;

            EXCEPTION WHEN OTHERS THEN
                 RAISE WARNING 'create cycle %', sqlerrm;
                 EXIT; -- #21113,#21059 Vermeidung von Endlosschleifen
            END;

            _cyclesrec.cyc_nexterm := cyc_nexterm FROM cycles WHERE cyc_id = _cyclesrec.cyc_id;

        END LOOP;

    END LOOP;

    RETURN;
  END $$ LANGUAGE plpgsql;
--

-- manuelles Auslösen
CREATE OR REPLACE FUNCTION cycles__execute__manual(inSrcDbrid varchar(32), inSrcKey varchar(100), inTrg_modul varchar(100), inManualDate date) RETURNS VOID AS $$
    DECLARE cyclesrec record;
            today date;
    BEGIN
        IF inSrcKey IN ('vertrag_pos', 'ab2', 'abk') THEN

            IF inSrcKey = 'vertrag_pos' THEN
                PERFORM cycle__execute__vertrag_pos(inSrcDbrid, inManualDate);
            END IF;

            IF inSrcKey = 'ab2' THEN
                PERFORM cycle__execute__ab2(inSrcDbrid, inManualDate);
            END IF;

            IF inSrcKey = 'abk' THEN
                PERFORM cycle__execute__abk(inSrcDbrid, inTrg_modul, inManualDate);
            END IF;

            UPDATE cycles SET
                   cyc_lasterm = inManualDate,
                   -- nächster Termin hinter manuellem Termin, wenn >= heute (ansonsten kann es 2x dieselbe Aktion geben)
                   cyc_nexterm = cycles__date_next__by__cycle_dbrid(cyc_begin, cyc_end, inManualDate, cyc_interval, cyc_multi, inSrcDbrid)
             WHERE cyc_src_dbrid = inSrcDbrid
               AND cyc_src_key = inSrcKey;

            IF NOT FOUND THEN
                INSERT INTO cycles (cyc_src_dbrid, cyc_src_key, cyc_trg_table, cyc_lasterm)
                SELECT inSrcDbrid, inSrcKey, inTrg_modul, inManualDate;
            END IF;
        END IF;

    RETURN;
  END $$ LANGUAGE plpgsql;
--

-- Vertragsbezogene Zyklus-Funktion
CREATE OR REPLACE FUNCTION cycle__execute__vertrag_pos(inDbrid varchar(32), inManualDate date DEFAULT NULL) RETURNS VOID AS $$
    DECLARE vertragposrec record;
    BEGIN
        SELECT vtp_type, vtp_id INTO vertragposrec
          FROM vertrag_pos
         WHERE vertrag_pos.dbrid = inDbrid;

        IF vertragposrec.vtp_type = 'A' THEN
           PERFORM TWawi.auftg__from__vertrag_pos__cycles__create(vertragposrec.vtp_id, inManualDate);
        END IF;

        IF vertragposrec.vtp_type = 'E' THEN
           PERFORM TWawi.ldsdok__from__vertrag_pos__cycles__create(vertragposrec.vtp_id, inManualDate);
        END IF;

        RETURN;
    END $$ LANGUAGE plpgsql;
--

-- Erstellung Auftrag vertrag_pos per Zyklus
CREATE OR REPLACE FUNCTION TWawi.auftg__from__vertrag_pos__cycles__create(
    IN _VertragPosID integer,
    IN _ManualDate   date DEFAULT null
    )
    RETURNS integer
    AS $$
    DECLARE _vertragposrec record;
            _result        integer;
            nummernkreis   varchar;
    BEGIN
        SELECT vertrag_pos.*,
               coalesce(_ManualDate, cyc_nexterm) AS nexterm,
               vtr_krz, vtr_an_nr, vtr_nr_ext, vtr_apext_krzl, vtr_apext, vtr_apint,
               an_sortkrz, ak_ks, ak_zeko, ac_ks, ac_konto_erl,
               coalesce(a1_wuco, TSystem.Settings__GetInteger('auftgsteucode')) AS steucode, steu_proz
          INTO _vertragposrec
          FROM      vertrag_pos
          LEFT JOIN cycles  ON cyc_src_dbrid = vertrag_pos.dbrid AND cyc_src_key = 'vertrag_pos'
          JOIN      vertrag ON vtr_nr = vtp_vtr_nr
          LEFT JOIN anl     ON an_nr = vtr_an_nr
          JOIN      art     ON ak_nr = vtp_ak_nr
          JOIN      artcod  ON ac_n = ak_ac
          LEFT JOIN adk1    ON a1_krz = vtr_krz
          LEFT JOIN steutxt ON steu_z = coalesce(a1_wuco, TSystem.Settings__GetInteger('auftgsteucode'))
         WHERE vtp_id = _VertragPosID;

        IF _vertragposrec.vtp_id IS NOT NULL THEN
          nummernkreis = TSystem.Settings__Get('Vertrag.ext_auftg.numkr');
          IF nummernkreis = '' OR nummernkreis IS null THEN
              nummernkreis = 'vertrag';
          END IF;

          INSERT INTO auftg (ag_astat,
                             ag_nr,
                             ag_vtp_id,
                             ag_lkn,
                             ag_krzl,
                             ag_krzf,
                             ag_kontakt,
                             ag_aknr, ag_stk, ag_mcv,
                             ag_vkp, ag_arab,
                             ag_kdatum,
                             ag_postxt, ag_txt,
                             ag_prkl, ag_kukl,
                             ag_an_nr,
                             ag_dispokrzl, ag_dispo,
                             ag_ks,
                             ag_konto,
                             ag_bstat,
                             ag_steucode, ag_ustpr
                             )
                      SELECT 'E',
                             CASE nummernkreis -- NICHT mit IFTHEN: führt nämlich beide Ergebnisse vorher aus. Damit immer "Lücke gefunden"/Reservierungen.
                                 WHEN 'vertrag' THEN 'VA.'||_vertragposrec.vtp_vtr_nr
                                 ELSE getnumcirclenr(nummernkreis)
                             END,
                             _vertragposrec.vtp_id,
                             _vertragposrec.vtr_krz, -- Bestelleradresse ag_lkn aus Vertragskopf
                             coalesce(_vertragposrec.vtp_address, _vertragposrec.an_sortkrz, _vertragposrec.vtr_krz), -- Lieferadresse ag_krzl entweder aus Vertragspos, Projekt Standort anl.an_sortkrz oder Vertragskopf
                             _vertragposrec.vtr_krz, -- Rechnungsadresse ag_krzf aus Vertragskopf
                             coalesce(_vertragposrec.vtp_apint, _vertragposrec.vtr_apint),
                             _vertragposrec.vtp_ak_nr, _vertragposrec.vtp_qty, _vertragposrec.vtp_mce,
                             _vertragposrec.vtp_price, _vertragposrec.vtp_rabatt,
                             _vertragposrec.nexterm,
                             _vertragposrec.vtp_txt, _vertragposrec.vtp_dokutxt,
                             1, 1,
                             _vertragposrec.vtr_an_nr,
                             _vertragposrec.vtr_apext_krzl, _vertragposrec.vtr_apext,
                             coalesce(_vertragposrec.vtp_ks, _vertragposrec.ak_ks, _vertragposrec.ac_ks),
                             coalesce(_vertragposrec.vtp_konto, _vertragposrec.ak_zeko, _vertragposrec.ac_konto_erl),
                             _vertragposrec.vtp_bstat,
                             _vertragposrec.steucode, _vertragposrec.steu_proz

          RETURNING ag_id INTO _result;
        END IF;

        RETURN _result;
    END $$ LANGUAGE plpgsql;

--

-- Erstellung Einkauf vertrag_pos per Zyklus
CREATE OR REPLACE FUNCTION TWawi.ldsdok__from__vertrag_pos__cycles__create(
    IN inVertragPosID integer,
    IN inManualDate   date DEFAULT NULL
    )
    RETURNS VOID
    AS $$
    DECLARE vertragposrec record;
            nummernkreis  varchar;
    BEGIN
        SELECT vertrag_pos.*, coalesce(inManualDate, cyc_nexterm) AS nexterm, vtr_krz, vtr_an_nr, vtr_nr_ext, vtr_apext_krzl, vtr_apext, vtr_apint,
               ak_ks, ak_awko, ac_ks, ac_konto, coalesce(a2_wuco, TSystem.Settings__GetInteger('einksteucode')) AS steucode, steu_proz
          INTO vertragposrec
          FROM vertrag_pos
          LEFT JOIN cycles ON cyc_src_dbrid = vertrag_pos.dbrid AND cyc_src_key = 'vertrag_pos'
          JOIN vertrag ON vtr_nr = vtp_vtr_nr
          JOIN art ON ak_nr = vtp_ak_nr
          JOIN artcod ON ac_n = ak_ac
          LEFT JOIN adk2 ON a2_krz = vtr_krz
          LEFT JOIN steutxt ON steu_z = coalesce(a2_wuco, TSystem.Settings__GetInteger('einksteucode'))
         WHERE vtp_id = inVertragPosID;

        IF vertragposrec.vtp_id IS NOT NULL THEN
            nummernkreis = TSystem.Settings__Get('Vertrag.int_ldsdok.numkr');
            IF nummernkreis = '' OR nummernkreis IS null THEN
                nummernkreis = 'vertrag';
            END IF;

            INSERT INTO ldsdok (ld_code,
                                ld_auftg,
                                ld_vtp_id,
                                ld_kn,
                                ld_krzl,
                                ld_kontakt,
                                ld_aknr, ld_stk, ld_mce,
                                ld_ep, ld_arab,
                                ld_term,
                                ld_txtint, ld_txt,
                                ld_an_nr,
                                ld_lkontaktkrzl, ld_lkontakt,
                                ld_ks,
                                ld_konto,
                                ld_bstat,
                                ld_waer,
                                ld_steucode, ld_steuproz)
                         SELECT 'I',
                                CASE nummernkreis -- NICHT mit IFTHEN: führt nämlich beide Ergebnisse vorher aus. Damit immer "Lücke gefunden"/Reservierungen.
                                    WHEN 'vertrag' THEN 'VE.'||vertragposrec.vtp_vtr_nr
                                    ELSE getnumcirclenr(nummernkreis)
                                END,
                                vertragposrec.vtp_id,
                                '#',
                                coalesce(vertragposrec.vtp_address, '#'),
                                coalesce(vertragposrec.vtp_apint, vertragposrec.vtr_apint),
                                vertragposrec.vtp_ak_nr, vertragposrec.vtp_qty, vertragposrec.vtp_mce,
                                vertragposrec.vtp_price, vertragposrec.vtp_rabatt,
                                vertragposrec.nexterm,
                                vertragposrec.vtp_txt, vertragposrec.vtp_dokutxt,
                                vertragposrec.vtr_an_nr,
                                vertragposrec.vtr_apext_krzl, vertragposrec.vtr_apext,
                                coalesce(vertragposrec.vtp_ks, vertragposrec.ak_ks, vertragposrec.ac_ks),
                                coalesce(vertragposrec.vtp_konto, vertragposrec.ak_awko, vertragposrec.ac_konto),
                                vertragposrec.vtp_bstat,
                                TSystem.Settings__Get('BASIS_W'),
                                vertragposrec.steucode, vertragposrec.steu_proz
                         ;
        END IF;

        RETURN;
    END $$ LANGUAGE plpgsql;
--

SELECT TSystem.Settings__Set('copyABKwithResultValues', 'F');  -- Einstellung ob bei der ABK-Kopie die Werte der Übergabeparameter mitgegeben werden sollen.
                                                    -- Fall Sunstrom: ABK in Service kopieren, Inhalte über Rückgabewerte, hier also TRUE

-- Erstellung ABK-Struktur per Zyklus mit Zielmodul
CREATE OR REPLACE FUNCTION cycle__execute__abk(inABKdbrid varchar, inTrg_modul varchar(100), inManualDate date DEFAULT NULL) RETURNS VOID AS $$
  DECLARE trg_data record; -- AS dbrid, AS tablename, AS keyvalue
          src_abk record;
          trg_kanf record;
  BEGIN
    SELECT ab_ix, coalesce(ab_tablename,'') AS ab_tablename, coalesce(ab_keyvalue,'') AS ab_keyvalue, ab_an_nr, ab_ap_nr, coalesce(ab_ap_bem, ak_bez, '') AS ab_ap_bem
      INTO src_abk
      FROM abk
      LEFT JOIN art ON ak_nr = ab_ap_nr
     WHERE abk.dbrid = inABKdbrid;

    IF src_abk.ab_ix IS NOT NULL THEN
        IF inTrg_modul = 'qabservice' THEN
            -- Serviceanfrage erzeugen. Wird benötigt für Projektnummer, Fehlerbeschreibung
            INSERT INTO kundanfrage (kanf_nr, kanf_art, kanf_date, kanf_via, kanf_isservice, kanf_an_nr, kanf_txt, kanf_done)
                SELECT getnumcirclenr('kanfrageser'), 'IS', current_date,
                lang_text(16258) || ', ABK: ' || src_abk.ab_ix || ', ' || lang_text(15429) || ': ' || src_abk.ab_tablename || ' ' || src_abk.ab_keyvalue, -- Wartungszyklus ... Identifikator
                true, src_abk.ab_an_nr, src_abk.ab_ap_nr || ' ' || src_abk.ab_ap_bem, true
            RETURNING kanf_nr INTO trg_kanf;

            -- Serviceanfrage-Position erzeugen. Für Verlinkung mit Servicevorfall
            INSERT INTO kundservicepos (kanfse_kanf_nr, kanfse_pos, kanfse_sfall)
                SELECT trg_kanf.kanf_nr, 10,
                coalesce(src_abk.ab_ap_nr || ' ' || src_abk.ab_ap_bem,
                lang_text(16258) || ', ABK: ' || src_abk.ab_ix || ', ' || lang_text(15429) || ': ' || src_abk.ab_tablename || ' ' || src_abk.ab_keyvalue)
            RETURNING kanfse_id INTO trg_kanf;

            -- eigtl. Servicevorfall
            INSERT INTO qab(q_nr, q_typ, q_ab_ix, q_dat, q_isservice, q_kanfse_id, q_fehtxt, q_feh_id, q_ent_id)
                SELECT getnumcirclenr('qabser')::integer, 7, src_abk.ab_ix, coalesce(inManualDate, current_date), true, trg_kanf.kanfse_id, src_abk.ab_ap_nr || ' ' || src_abk.ab_ap_bem,
                132, 20 -- Kein Fehler, Rechnung
            -- Daten für copyABK_toDbrid definieren
            RETURNING dbrid::varchar AS dbrid, 'qab'::varchar AS tablename, q_nr::varchar AS keyvalue INTO trg_data;

        ELSIF inTrg_modul IS NULL THEN -- in dasselbe Modul
            -- Daten für copyABK_toDbrid definieren
            SELECT ab_dbrid AS dbrid, ab_tablename AS tablename, ab_keyvalue AS keyvalue INTO trg_data FROM abk WHERE abk.dbrid = inABKdbrid;
        ELSE
            RAISE EXCEPTION 'Undefined target key for abk specified: %', inTrg_modul;
        END IF;

        PERFORM copyABK_toDbrid(trg_data.dbrid, trg_data.tablename, trg_data.keyvalue, src_abk.ab_ix, true, TSystem.Settings__GetBool('copyABKwithResultValues')); -- inkl. Material und Rückgabewerten anhand Setting

    END IF;

    RETURN;
END $$ LANGUAGE plpgsql;
--

-- Erstellung Arbeitsgang per Zyklus
CREATE OR REPLACE FUNCTION cycle__execute__ab2(inAB2dbrid varchar, inManualDate date DEFAULT NULL) RETURNS VOID AS $$
    DECLARE ab2rec RECORD;
          newab2id integer;
    BEGIN
        SELECT ab2.a2_ab_ix, ab2.dbrid, ab2.a2_dlz, coalesce(inManualDate, cyc_nexterm) AS nexterm
          INTO ab2rec
          FROM ab2
          LEFT JOIN cycles ON cyc_src_dbrid = inAB2dbrid AND cyc_src_key = 'ab2'
         WHERE ab2.dbrid = inAB2dbrid;

        IF ab2rec.dbrid IS NOT NULL THEN
           PERFORM copyAG_toABK(ab2rec.a2_ab_ix, NULL, ab2rec.dbrid, false, NULL, NULL, ab2rec.nexterm, (ab2rec.nexterm + ab2rec.a2_dlz * '1 day'::interval));
        END IF;

        RETURN;
    END $$ LANGUAGE plpgsql;
--

-- ?: a2_o2_id, a2_ncnr, a2_auswpruef(false da neu), a2_interm(true, da term),
-- a2_timechecked, a2_ta, a2_group_ta, a2_time_stemp


-- keine leeren Statements am Ende vom Erstellen der DB erlaubt.
SELECT TRUE;
